<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database connection
$host = "localhost";
$user = "cmhost_tour";
$pass = "cmhost_tour";
$db   = "cmhost_tour";
$conn = mysqli_connect($host, $user, $pass, $db);

if (!$conn) {
    echo json_encode(['status' => 'error', 'message' => 'Database connection failed']);
    exit;
}

// Load Composer autoload
require 'vendor/autoload.php';
use Google\Auth\Credentials\ServiceAccountCredentials;

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Only POST method allowed');
    }

    // Get form data
    $notification_title = $_POST['notification_title'] ?? '';
    $notification_msg   = $_POST['notification_msg'] ?? '';
    $image_url          = $_POST['image_url'] ?? '';
    $send_type          = $_POST['send_type'] ?? '';
    $selected_users     = $_POST['selected_users'] ?? [];

    // Validate required fields
    if (empty($notification_title) || empty($notification_msg) || empty($send_type)) {
        throw new Exception('Title, message, and send type are required');
    }

    // Validate image URL if provided
    if (!empty($image_url) && !filter_var($image_url, FILTER_VALIDATE_URL)) {
        throw new Exception('Invalid image URL format');
    }

    // FCM setup
    $serviceAccountPath = __DIR__ . '/src/serviceAccountKey.json';
    if (!file_exists($serviceAccountPath)) {
        throw new Exception('Service account file not found');
    }

    $json = json_decode(file_get_contents($serviceAccountPath), true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid service account JSON');
    }

    $projectId = 'ng-tournament-01';
    $scopes = ['https://www.googleapis.com/auth/firebase.messaging'];
    $credentials = new ServiceAccountCredentials($scopes, $json);
    $authTokenArray = $credentials->fetchAuthToken();
    $accessToken = $authTokenArray['access_token'] ?? null;

    if (!$accessToken) {
        throw new Exception('Failed to obtain FCM access token');
    }

    // Prepare FCM message
    $message = [
        'notification' => [
            'title' => $notification_title,
            'body'  => $notification_msg
        ],
        'android' => [
            'notification' => [
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK'
            ]
        ]
    ];

    if ($image_url) {
        $message['notification']['image'] = $image_url;
    }

    $success_count = 0;
    $failure_count = 0;
    $total_sent = 0;

    // Send notification based on type
    if ($send_type === 'all') {
        // Send to all users via topic
        $body = ['message' => array_merge($message, ['topic' => 'all'])];
        $result = sendFCMNotification($body, $accessToken, $projectId);
        
        if ($result['success']) {
            $success_count = 1;
            $total_sent = 1;
            
            // Get total user count for logging
            $count_query = "SELECT COUNT(*) as total FROM user_details WHERE status = 'Active'";
            $count_result = mysqli_query($conn, $count_query);
            $count_row = mysqli_fetch_assoc($count_result);
            $user_count = $count_row['total'];
            
            // Save notification to database (commented out as in original)
            // saveNotificationToDb($conn, $notification_title, $notification_msg, $image_url, null, 'all', null, $user_count);
            
        } else {
            $failure_count = 1;
            throw new Exception('Failed to send notification to all users: ' . $result['error']);
        }
        
    } elseif ($send_type === 'single' || $send_type === 'multiple') {
        if (empty($selected_users)) {
            throw new Exception('Please select at least one user');
        }

        // Ensure selected_users is array for single user too
        if (!is_array($selected_users)) {
            $selected_users = [$selected_users];
        }

        // Get FCM tokens for selected users
        $user_ids = implode(',', array_map('intval', $selected_users));
        $tokens_query = "SELECT id, fcm_token, CONCAT(fname, ' ', lname) as full_name 
                        FROM user_details 
                        WHERE id IN ($user_ids) AND fcm_token IS NOT NULL AND fcm_token != ''";
        $tokens_result = mysqli_query($conn, $tokens_query);

        if (!$tokens_result || mysqli_num_rows($tokens_result) === 0) {
            throw new Exception('No valid FCM tokens found for selected users');
        }

        // Send notification to each user individually
        while ($user = mysqli_fetch_assoc($tokens_result)) {
            $body = ['message' => array_merge($message, ['token' => $user['fcm_token']])];
            $result = sendFCMNotification($body, $accessToken, $projectId);
            
            $total_sent++;
            if ($result['success']) {
                $success_count++;
            } else {
                $failure_count++;
                error_log("Failed to send notification to user {$user['full_name']} (ID: {$user['id']}): {$result['error']}");
            }
        }

        // Save notification to database (commented out as in original)
        // saveNotificationToDb($conn, $notification_title, $notification_msg, $image_url, null, $send_type, $selected_users, $success_count);
    }

    // Prepare response
    if ($success_count > 0) {
        $message = "Notification sent successfully! ";
        if ($total_sent > 1) {
            $message .= "Success: $success_count, Failed: $failure_count out of $total_sent";
        }
        echo json_encode(['status' => 'success', 'message' => $message]);
    } else {
        throw new Exception('All notification attempts failed');
    }

} catch (Exception $e) {
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
} catch (Error $e) {
    echo json_encode(['status' => 'error', 'message' => 'System error: ' . $e->getMessage()]);
}

// Function to send FCM notification
function sendFCMNotification($body, $accessToken, $projectId) {
    $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";
    $headers = [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json; UTF-8',
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($response === false) {
        return ['success' => false, 'error' => 'CURL Error: ' . $error];
    }

    if ($httpCode === 200) {
        return ['success' => true, 'response' => $response];
    } else {
        $errorResponse = json_decode($response, true);
        $errorMsg = isset($errorResponse['error']['message']) ? $errorResponse['error']['message'] : 'HTTP Error ' . $httpCode;
        return ['success' => false, 'error' => $errorMsg];
    }
}
?>